import os
import re
from datetime import datetime

def contains_chinese(text):
    """
    检查字符串是否包含中文字符。
    Unicode 范围 \u4e00-\u9fff 涵盖了大部分 CJK 统一表意文字。
    """
    return re.search(r'[\u4e00-\u9fff]', text) is not None

def process_txt_file(input_filepath):
    """
    读取指定文件，过滤掉不含中文字符的行，并返回包含中文字符的行列表。
    """
    chinese_lines = []
    total_lines = 0
    removed_lines = 0

    print(f"开始处理文件: {input_filepath}")
    
    try:
        with open(input_filepath, 'r', encoding='utf-8') as f:
            for line_num, line in enumerate(f, 1):
                total_lines += 1
                if contains_chinese(line):
                    chinese_lines.append(line)
                else:
                    removed_lines += 1
                    # 可以选择打印被移除的行，如果需要调试
                    # print(f"  移除行 {line_num}: {line.strip()}")
        
        print(f"  原始文件总行数: {total_lines}")
        print(f"  包含中文字符的行数: {len(chinese_lines)}")
        print(f"  已移除不含中文字符的行数: {removed_lines}")
        
        return chinese_lines

    except FileNotFoundError:
        print(f"❌ 错误: 文件 '{input_filepath}' 不存在。")
        return None
    except UnicodeDecodeError:
        print(f"❌ 错误: 无法以 UTF-8 编码读取文件 '{input_filepath}'。请检查文件编码。")
        return None
    except Exception as e:
        print(f"❌ 错误: 读取文件 '{input_filepath}' 时发生未知错误: {e}")
        return None

def save_filtered_content(output_filepath, filtered_lines):
    """
    将过滤后的内容保存到新的文件中。
    """
    try:
        with open(output_filepath, 'w', encoding='utf-8') as f:
            for line in filtered_lines:
                f.write(line)
        print(f"✅ 过滤后的内容已成功保存到: {output_filepath}")
        return True
    except Exception as e:
        print(f"❌ 错误: 写入文件 '{output_filepath}' 失败: {e}")
        return False

def main():
    print("--- 文本文件中文行筛选工具 ---")
    print("⚠️ 警告：此操作会筛选您的文本内容。在运行前，请务必备份您的原始文件！")
    
    # 获取用户输入的txt文件路径
    # .strip('"').strip("'") 用于移除用户可能输入的引号
    input_file = input("请输入要处理的 .txt 文件路径（例如 D:\\MyTexts\\document.txt）：").strip('"').strip("'")

    # 检查输入的路径是否是一个有效的txt文件
    if not os.path.isfile(input_file) or not input_file.lower().endswith('.txt'):
        print(f"❌ 错误: '{input_file}' 不是一个有效的 .txt 文件路径。请检查。")
        input("\n按回车键退出程序...")
        return

    # 处理文件，获取包含中文字符的行
    filtered_lines = process_txt_file(input_file)
    
    if filtered_lines is None: # 表示处理过程中有错误发生
        input("\n按回车键退出程序...")
        return
        
    if not filtered_lines:
        print("ℹ️ 文件中未找到任何包含中文字符的行。未生成新文件。")
        input("\n按回车键退出程序...")
        return

    # 生成输出文件名：在原文件名前添加 '_chinese_only' 和时间戳
    base_name = os.path.basename(input_file)
    dir_name = os.path.dirname(input_file)
    name_without_ext, ext = os.path.splitext(base_name)
    
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    output_filename = f"{name_without_ext}_chinese_only_{timestamp}{ext}"
    output_filepath = os.path.join(dir_name, output_filename)

    # 保存过滤后的内容
    save_filtered_content(output_filepath, filtered_lines)
    
    print("\n------------------------")
    
    # 保持窗口打开，直到用户按下回车键
    input("\n按回车键退出程序...")

if __name__ == "__main__":
    main()
